/*
 * LIBFDMS: FDMS Library
 * Copyright (C) 2008 Jon Siddle <jon@trapdoor.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/** \file */

#ifndef _FDMS_H_
#define _FDMS_H_

#include <sys/types.h>
#include <inttypes.h>

#if defined(_WIN32)
  #if defined(fdms_EXPORTS)
    #define FDMS_EXPORT __declspec(dllexport)
  #else
    #define FDMS_EXPORT __declspec(dllimport)
  #endif
#else
  #define FDMS_EXPORT
#endif

/** Size of a program header */
#define PRG_HEADER_SIZE (0x200LL*8)

#ifdef __cplusplus
extern "C" {
#endif

    /** Opaque FDMS disk handle */
    typedef struct _fdms_vol fdms_vol;
    /** Opaque FDMS program handle */
    typedef struct _fdms_prg fdms_prg;
    /** Opaque FDMS track handle */
    typedef struct _fdms_trk fdms_trk;
    /** Opaque FDMS segment handle */
    typedef struct _fdms_seg fdms_seg;

    /**
     * Midi sync setting values.
     */
    enum midi_sync {
	fdms_sync_off,	/**< OFF */
	fdms_sync_clk,	/**< CLK */
	fdms_sync_mtc	/**< MTC */
    };

    /**
     * Offset mode setting values.
     */
    enum offset_mode {
	fdms_abs,	/**< ABS */
	fdms_bar	/**< BAR */
    };

    /* Information structs.
     * These are returned by get_* functions
     */

    /**
     * Information about an fdms program.
     */
    struct fdms_prg_info {
	int preroll;			/**< preroll */
	int metronome;			/**< metronome on/off */
	enum midi_sync midi_sync_out;   /**< midi sync = off/clk/mtc */
	int mtc_frame_rate;		/**< frame rate = 24/25/30 */
	int mtc_off_h;			/**< mtc offset (hours) */
	int mtc_off_m;			/**< mtc offset (minutes) */
	int mtc_off_s;			/**< mtc offset (seconds) */
	int mtc_off_f;			/**< mtc offset (frames) */
	int mtc_off_u;			/**< mtc offset (sub-frames) */
	enum offset_mode offset_mode;   /**< offset mode = abs/bar */
	int record_protect;		/**< record protect on/off */
	int offset;			/**< offset */

	int sample_size;		/**< Sample size */
	int sample_rate;		/**< Sample rate */
    };

    /**
     * Entry in tempo map.
     */
    struct fdms_tempo {
	int bar;	    /**< Bar to which tempo applies */
	int y;		    /**< ?? */
	int tempo;	    /**< Tempo */
    };

    /**
     * Entry in signature map
     */
    struct fdms_sig {
	int bar;	    /**< Bar to which signature applies */
	int beats;	    /**< Beats in a bar */
	int value;	    /**< Value of said beats */
    };

    /**
     * Open volume.
     *
     * path should point to an image or device.
     *
     * Opens the device or image path, and stores a volume handle in pp_vol.
     * This handle must be passed to fdms_close() once it is no longer needed.
     *
     * \param path the path to an FDMS image or device.
     * \param pp_vol the volume handle will be stored here.
     */
    int fdms_open(const char *path, fdms_vol **pp_vol);

    /**
     * Get the program capacity.
     *
     * This is the number of possible programs for this filesystem.
     * Usually 128 (limited to 99 by the device).
     *
     * \param vol the volume handle.
     */
    int fdms_get_prg_capacity(fdms_vol *vol);

    /**
     * Get program count.
     *
     * Returns the actual number of valid programs in the volume.
     *
     * \param vol the volume handle.
     */
    int fdms_get_prg_count(fdms_vol *vol);

    /**
     * Open program.
     *
     * The program handle must be passed to fdms_close_prg() once it is no
     * longer needed.
     *
     * \param vol the volume handle.
     * \param p the 0-based program index
     * \param pp_prg the program handle will be stored here.
     */
    int fdms_open_prg(fdms_vol *vol, int p, fdms_prg **pp_prg);

    /**
     * Get track cacity.
     *
     * Returns the number of tracks this program can hold. Usually 24,
     * limited by the device to whatever it can handle.
     *
     * \param prg the volume handle.
     */
    int fdms_get_trk_capacity(fdms_prg *prg);
    
    /**
     * Get track count.
     *
     * Returns the number of actual tracks in the program.
     *
     * \param prg the volume handle.
     */
    int fdms_get_trk_count(fdms_prg *prg);

    /**
     * Open track.
     *
     * The track handle must be passed to fdms_close_trk() once it is no longer
     * needed.
     *
     * \param prg the program handle.
     * \param t the 0-based track index
     * \param pp_trk the track handle will be stored here.
     */
    int fdms_get_trk(fdms_prg *prg, int t, fdms_trk **pp_trk);

    /**
     * Get track length (bytes).
     *
     * \param trk the track handle.
     */
    size_t fdms_get_trk_length(fdms_trk *trk);

    /* Low level segment interface */
    
    /**
     * Get segment count.
     *
     * A track is made up of logically consecutive segments numbered 0..n-1.
     * This returns n.
     *
     * \param trk the track handle.
     */
    int fdms_get_seg_count(fdms_trk *trk);

    /**
     * Get segment length (bytes) of segment s.
     *
     * \param trk the track handle.
     * \param s the 0-based segment index.
     */
    size_t fdms_get_seg_length(fdms_trk *trk, int s);

    /**
     * Read a maximum of count bytes from segment s starting at offset.
     * Return the number of bytes actually read.
     */
    int fdms_read_seg_audio(char *buffer, fdms_trk *trk, int s, off_t offset, size_t count);

    /* END Low level segment interface */

    /**
     * Read a maximum of count bytes from all segments of trk starting at offset.
     * Return the number of bytes actually read.
     */
    int fdms_read_trk_audio(char *buffer, fdms_trk *trk, off_t offset, size_t count);

    /**
     * Close fdms and release resources.
     */
    int fdms_close(fdms_vol *vol);

    /**
     * Return the program name.
     */
    const char *fdms_get_prg_name(fdms_prg *prg);

    /**
     * Return program information.
     */
    int fdms_get_prg_info(fdms_prg *prg, struct fdms_prg_info *prg_info);

    /**
     * Return tempo map.
     */
    int fdms_get_tempo_map(fdms_prg *prg, struct fdms_tempo *tempo_map);

    /**
     * Return time sig map.
     */
    int fdms_get_timesig_map(fdms_prg *prg, struct fdms_sig *sig_map);

    /**
     * Return the raw header in the (caller-allocated) buffer header.
     *
     * This method takes fdms_vol* rather than fdms_prg* in case fdms_open_prg
     * can't parse the program header
     */
    int fdms_get_prg_header_raw(fdms_vol *vol, int p, char *header);

    /**
     * Return the raw tempo map in the (caller-allocated) buffer tempo_map.
     *
     * This method takes fdms_vol* rather than fdms_prg* in case fdms_open_prg
     * can't parse the program header
     */
    int fdms_get_tempo_map_raw(fdms_vol *vol, int p, char *tempo_map);

    /**
     * Return the raw time sig map in the (caller-allocated) buffer timesig_map.
     *
     * This method takes fdms_vol* rather than fdms_prg* in case fdms_open_prg
     * can't parse the program header
     */
    int fdms_get_timesig_map_raw(fdms_vol *vol, int p, char *timesig_map);

#ifdef __cplusplus
}
#endif

#endif /* _FDMS_H_ */
