/*
 * Java properties library
 * Copyright (C) 2003 Jon Siddle <jon@trapdoor.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.trapdoor.properties;

import cojen.classfile.CodeBuilder;

/**
 * A Class property defines a conceptual variable
 * related to a particular class.
 * The property may be read-write, read-only, or
 * write-only. It may relate directly to the the
 * class (static) or to instances of the class
 * (non-static).
 * This implementation is immutable (as should be
 * all subclasses).
 */
public abstract class ClassProperty {
	/*{{{Access modes*/
	public static final int CP_RO = 0;
	public static final int CP_WO = 1;
	public static final int CP_RW = 2;
	/*}}}*/

	/**How this property should deal with nulls*/
	private boolean isDynamicNullHandling;

	/**The class to which this property belongs*/
	private Class declaringClass;

	/**The name of this property*/
	private String propertyName;

	/**The access mode for this property*/
	private int access;

	/**The class of all property values*/
	private Class propertyType;

	/**Whether or not the property is static*/
	private boolean staticProperty;

	/**Default constructor simply saves class/property name*/
	ClassProperty(Class declaringClass, String propertyName, int access, Class propertyType, boolean staticProperty, boolean isDynamicNullHandling) {
		this.declaringClass=declaringClass;
		this.propertyName=propertyName;
		this.access=access;
		this.propertyType=propertyType;
		this.staticProperty=staticProperty;
		this.isDynamicNullHandling=isDynamicNullHandling;
	}

	/*Assume params are on stack in order*/
	protected abstract void compileGet( CodeBuilder b );
	protected abstract void compileSet( CodeBuilder b );
	protected abstract void compileGetTarget( CodeBuilder b );

	/**
	 * Get propertyName.
	 *
	 * @return propertyName as String.
	 */
	public String getPropertyName() {
		return propertyName;
	}

	/**
	 * Get Declaring class.
	 *
	 * @return Declaring class as Class.
	 */
	public Class getDeclaringClass() {
		return declaringClass;
	}

	/**
	 * Get property access mode.
	 *
	 * @return access mode.
	 */
	public int getAccessMode() {
		return access;
	}

	public boolean isDynamicNullHandling() {
		return isDynamicNullHandling;
	}

	/**
	 * Get property type.
	 *
	 * @return propertyType as Class.
	 */
	public Class getPropertyType() {
		return propertyType;
	}

	/**
	 * Get static modifier.
	 */
	public boolean isStatic() {
		return staticProperty;
	}

	/*{{{Value manipulation*/
	/**
	 * Set the value of this property in target to value.
	 * If this property is static, target must be null.
	 * If this property is non-static, target must be non-null.
	 * This is a convenience method for the common case where
	 * the set method takes a single argument.
	 */
	public abstract void setValue(Object target, Object value) throws IllegalArgumentException;

	public abstract Object getValue(Object source);
	/*}}}*/

	public abstract Object getTarget(Object target);
}


