/*
 * Java properties library
 * Copyright (C) 2003 Jon Siddle <jon@trapdoor.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.trapdoor.properties;

/**
 * A Property identifying key used to index the cache.
 * An instance of ClassPropertyKey is unique to a
 * class-propertyname combination. It is used in the
 * property cache to match property requests.
 * This class is immutable.
 */
public class ClassPropertyKey {
	/**Give property names class scope so they only have to be
	   unique to a particular class.*/
	private Class declaringClass;
	/**Property name*/
	private String propertyName;
	/**Null handling mode*/
	private boolean isDynamicNullHandling;

	public ClassPropertyKey(Class declaringClass, String propertyName, boolean isDynamicNullHandling) {
		this.declaringClass=declaringClass;
		this.propertyName=propertyName;
		this.isDynamicNullHandling=isDynamicNullHandling;
	}
	public ClassPropertyKey(ClassProperty property) {
		this(property.getDeclaringClass(),
			 property.getPropertyName(),
			 property.isDynamicNullHandling());
	}

	/**Overridden to depend only on class and name*/
	public int hashCode() {
		/*NB. we simulate Boolean.hashCode() here (for no particulary good reason)*/
		return 31*(isDynamicNullHandling?1231:1237)*declaringClass.hashCode()*13+propertyName.hashCode();
	}

	/**Overridden to depend only on class, name and mode*/
	public boolean equals(Object o) {
		ClassPropertyKey key = (ClassPropertyKey)o;
		return
			/*Class instances are unique, so == is ok*/
			(key.declaringClass==declaringClass) &&
			( ((key.propertyName==null) && (propertyName==null)) || key.propertyName.equals(propertyName) ) &&
			( key.isDynamicNullHandling == isDynamicNullHandling );
	}

	/**Return the declaring class*/
	public Class getDeclaringClass() {
		return declaringClass;
	}

	/**Return the property name*/
	public String getPropertyName() {
		return propertyName;
	}

	/**Returns the null handling mode*/
	public boolean isDynamicNullHandling() {
		return isDynamicNullHandling;
	}

	public String toString() {
		String className = getDeclaringClass().getName();
		int pkgIndex = className.lastIndexOf(".");
		if(pkgIndex>=0)className=className.substring(pkgIndex+1);

		return "["+className+"|"+getPropertyName()+"|"+isDynamicNullHandling()+"]";
	}
}
