/*
 * Java properties library
 * Copyright (C) 2003 Jon Siddle <jon@trapdoor.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.trapdoor.properties;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.HashMap;

import org.apache.log4j.Logger;

/**
 * This class manages instances of dynamically created classes.
 */
public abstract class DynamicClassManager {
	private static Logger logger = Logger.getLogger(DynamicClassManager.class);
	private static File path;
	private static ClassLoader classLoader;
	private static HashMap objectMap=new HashMap();

	/**
	 * New DCM which uses path to retrieve/store class definitions
	 * which are inner classes of parentClass.
	 */
	public static void init(File path) {
		DynamicClassManager.path=path;
	}

	/**
	 * Returns whether a dynamically generated object with the given key exists.
	 */
	public static boolean containsKey(Object key) {
		return objectMap.containsKey(key);
	}

	/**
	 * Return the object which is registered against key.
	 * If no object is found, return null.
	 */
	public static Object getObject(Object key) throws IOException {
		return objectMap.get(key);
	}

	/**
	 * Loads dynamic class className.
	 * This uses a separate classLoader (with the classloader of DynamicClassManager as the parent).
	 * This allows dynamic classes to be unloaded, by discarding this classloader.
	 * For this to work, the path must be somewhere NOT found by the parent classloader.
	 */
	public static void loadClass(String className) throws ClassNotFoundException {
		try {
			/* We initialize the classLoader the first time we need it.
			 * This gives us a chance to call init() with the path.
			 */
			if( classLoader==null ) {
				String classPathURL = "file:"+path.getPath()+"/";
				classLoader=new URLClassLoader(
						new URL[]{new URL(classPathURL)},
						DynamicClassManager.class.getClassLoader()
						);
			}
		} catch(MalformedURLException e) {
			throw new AssertionError(e);
		}

		try {
			Class c = Class.forName(className, true, classLoader);
		} catch(ClassNotFoundException e) {
			logger.error("Error Loading class: "+className+" from "+path.getPath());
			throw e;
		}
	}

	/**
	 * Returns the File corresponding to the generated .class file for the named class.
	 */
	public static File fileForClass(String className) {
		className=className.replace('.',File.separatorChar);
		File file = new File(path,className+".class");
		file.getParentFile().mkdirs();
		return file;
	}

	/**
	 * Forget all dynamically generated classes.
	 * This allows them to be re-loaded if necessary.
	 */
	public static void purge() {
		discardClassLoader();
		objectMap.clear();
	}

	/**
	 * Load all classes found in the path.
	 * This will NOT re-load already loaded classes. Call purge() first if this is desired.
	 */
	public static void load() throws ClassNotFoundException {
		load(path);
	}

	/**
	 * Load all classes found in path.
	 * If there is an error loading any classfile, it is deleted. This is to allow it
	 * to be re-generated on the next request to the manager.
	 */
	private static void load(File path) throws ClassNotFoundException {
		File files[] = path.listFiles();
		for(int i=0; files!=null && i<files.length; i++) {
			if(files[i].isDirectory()) load(files[i]);
			else {
				String name = files[i].getAbsolutePath();
				String prefix = DynamicClassManager.path.getAbsolutePath();

				int extIndex = name.indexOf(".class");
				if(extIndex<0) {
					logger.warn(name+" is not a .class file.");
				} else {
					try {
						loadClass(name.substring(prefix.length()+1,extIndex).replace(File.separatorChar,'.'));
					} catch(Throwable e) {
						logger.debug(e);
						logger.warn(name+" could not be loaded, deleting to allow regeneration.");
						if(!files[i].delete()) {
							logger.error(name+" could not be loaded, and could not be deleted.");
						}
					}
				}
			}
		}
	}

	/**
	 * Return the next available inner class id for outer class className.
	 * className must include the "$".
	 * Doesn't strictly have to be an outerclass, could just be a prefix.
	 */
	public static int nextId(String className) {
		int i=1;
		while(fileForClass(className+i).exists()) i++;
		return i;
	}

	/**
	 * Discard the current classloader.
	 * Guarantees that newly loaded classes will be re-loaded.
	 */
	public static void discardClassLoader() {
		classLoader=null;
	}

	/**
	 * Register an object with the associated key.
	 * Called by dynamically generated classes.
	 */
	public static void registerObject(Object key, Object o) {
		logger.debug("Registering "+o);
		objectMap.put(key,o);
	}

	/**
	 * Return the path for dynamic classes.
	 */
	public static File getPath() {
		return path;
	}
}
