/*
 * Java properties library
 * Copyright (C) 2003 Jon Siddle <jon@trapdoor.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.trapdoor.properties;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import cojen.classfile.*;

/** Nested class property.
 *  This ClassProperty implementation defines properties in terms of
 *  SimpleClassProperty s of objects returned by SimpleClassProperty s.
 *  That is a nested property defines a list of SimpleClassProperty s
 *  where each property relates to the class returned by the previous
 *  property.
 */
public class NestedClassProperty extends ClassProperty {
	List propertyList = new ArrayList();

	/**
	 * 
	 * @param name Property name.
	 */
	public NestedClassProperty(Class declaringClass, String propertyName, List propertyList, boolean isDynamicNullHandling)
	throws IllegalArgumentException/*{{{*/
	{
		super(declaringClass, propertyName, plAccess(propertyList), plType(propertyList), plStatic(propertyList), isDynamicNullHandling);
		this.propertyList=propertyList;
	}

	/*{{{Constructor helpers*/
	private static ClassProperty lastProperty( List propertyList ) {
		if( propertyList==null || propertyList.size()==0 )
			throw new IllegalArgumentException("null or zero length property list");
		return (ClassProperty) propertyList.get(propertyList.size()-1);
	}
	private static int plAccess( List propertyList ) {
		return lastProperty(propertyList).getAccessMode();
	}
	private static Class plType( List propertyList ) {
		return lastProperty(propertyList).getPropertyType();
	}
	private static boolean plStatic( List propertyList ) {
		return lastProperty(propertyList).isStatic();
	}
	/*}}}*/

	public static ClassProperty createProperty(ClassPropertyKey key) {
		return createProperty(key.getDeclaringClass(), key.getPropertyName(), key.isDynamicNullHandling());
	}
	/**
	 * Convenience method where the name defaults to the path.
	 */
	public static ClassProperty createProperty(Class declaringClass, String propertyName, boolean isDynamicNullHandling ) {
		return createProperty( declaringClass, propertyName, propertyName, isDynamicNullHandling );
	}

	/**
		 * Find a chain of SimpleClassProperty s which define a sequence of accessor methods,
	 * chained together by their return values, terminating in a final accessor OR mutator method.
	 */
	public static ClassProperty createProperty(Class declaringClass, String propertyName, String propertyPath, boolean isDynamicNullHandling ) {
		/*Split the property path around ";" separators*/
		String[] parts = propertyPath.split(":");

		if( parts.length<1 ) throw new IllegalArgumentException("Property path must have atleast 1 element");

		List propertyList = new ArrayList();

		Class currentClass = declaringClass;
		/*Iterate through the path, finding SimpleProperty s for
		  each, and updating (including casting) the next class*/
		for(int i=0; i<parts.length; ++i) {
			/*Check that previous property has read access*/
			if( i>0 && lastProperty(propertyList).getAccessMode()==CP_WO )
				throw new IllegalStateException("Write-Only properties must be last property in chain");

			/*Find cast prefix{{{*/
			Class castClass=null;
			int openCast, closeCast;
			openCast=parts[i].indexOf("(");
			closeCast=parts[i].indexOf(")");
			/*Check that both or neither (,) are present
			  and that ( is at index 0 if present*/
			if( ((openCast<0) ^ (closeCast<0)) || openCast>0 )
				throw new IllegalArgumentException("Illegal parenthesis");

			/*If there is no cast, check there is no ")"*/
			if(openCast<0) {
				if(closeCast>=0) throw new IllegalArgumentException("')' without '('");
			}
			/*If there is a cast, check validity and find and save cast class.*/
			else {
				/*'(' must be first char*/
				if(openCast>0) throw new IllegalArgumentException("cast must be a prefix");
				/*')' must be present. Don't need to check position since
				  '(' is first char and there can only be one set of parenthesis*/
				if(closeCast<0) throw new IllegalArgumentException("'(' without ')'");
				/*Check there is only one pair of parenthesis*/
				if( (parts[i].indexOf("(",openCast+1)>=0) || (parts[i].indexOf(")",closeCast+1)>=0) )
					throw new IllegalArgumentException("Illegal cast: '"+parts[i]+"'");

				/*Find class named in class string. Assuming it exists, save it
				  for next iteration's declaring class and process property suffix.*/
				try {
					castClass = Class.forName(parts[i].substring(1,closeCast));
					/*Remove cast string*/
					parts[i]=parts[i].substring(closeCast+1);
				} catch(ClassNotFoundException e) {
					throw new IllegalArgumentException("Illegal Cast");
				}
			}
			/*}}}*/

			/*Find "Simple" ClassProperty for this part of the path{{{
			  Try to find property of most refined class,
			  ascending class heirarchy until property is found.
			  Usually the simple property will actually be an instance
			  of SimpleClassProperty, but this is not required. The only
			  way another Nested property could be matched however, is
			  if a non-default propertyName (alias) is used.
			*/
			ClassProperty simpleProperty = null;
			Class ascendingClass = currentClass;
			while(ascendingClass!=null && simpleProperty==null) {
				try {
					simpleProperty = SimpleClassProperty.createProperty(ascendingClass, parts[i], isDynamicNullHandling);
				} catch(IllegalArgumentException e) {
					//e.printStackTrace();
					ascendingClass=ascendingClass.getSuperclass();
				} catch(IllegalStateException e) {
					//e.printStackTrace();
					ascendingClass=ascendingClass.getSuperclass();
				}
			}
			if( simpleProperty==null )
				throw new IllegalArgumentException("Could not resolve '"+propertyName+
												   "' on class '"+currentClass.getName()+
												   "' failed on property '"+parts[i] );

			/*Append this simple property to the list*/
			propertyList.add(simpleProperty);
			/*}}}*/

			/*The next simple property will belong to the class returned by
			  this simple property unless it is overridden by an explicit cast.*/
			currentClass = (castClass==null)?simpleProperty.getPropertyType():castClass;
		}

		return new NestedClassProperty(declaringClass, propertyName, propertyList, isDynamicNullHandling);
	}


	/**
	 * Create a new object with a null constructor.
	 * Illegal argument exception if no null constructor exists.
	 * (this is propagated out from setValue(.))
	 */
	private static Object instantiate(Class klass) throws IllegalArgumentException {
		try {
			Constructor constructor = klass.getConstructor(new Class[]{});
			return constructor.newInstance(null);
		} catch(Exception e) {
			IllegalArgumentException iae = new IllegalArgumentException("Failed dynamic null handling: cannot instantiate without a null constructor.");
			iae.initCause(e);
			throw(iae);
		}
	}

	protected void compileGet( CodeBuilder b ) {
		Iterator it = propertyList.iterator();
		while(it.hasNext()) {
			ClassProperty property = (ClassProperty)it.next();
			property.compileGet(b);
		}
	}

	protected void compileSet( CodeBuilder b ) {
		ClassProperty property;
		LocalVariable lastObject=null;

		b.swap();

		if(isDynamicNullHandling()) {
			lastObject = b.createLocalVariable("lastObject", TypeDesc.OBJECT);
			b.loadNull();
			b.storeLocal(lastObject);
		}

		Iterator it = propertyList.iterator();
		ClassProperty previousProperty=null;
		while(it.hasNext()) {
			property = (ClassProperty)it.next();

			if(isDynamicNullHandling()) {
				b.dup();
				Label notNull = b.createLabel();
				b.ifNullBranch(notNull,false);
				/*Remove null from stack*/
				b.pop();

				/*new Whatever()*/
				TypeDesc propertyTD = TypeDesc.forClass(property.getDeclaringClass());
				b.newObject(propertyTD);
				b.dup();
				b.invokeConstructor(propertyTD,null);

				/*TODO - validity if previousProperty==null?*/
				if(previousProperty!=null) {
					/*lastObject.setXXX(newObject)*/
					b.dup();
					b.loadLocal(lastObject);
					b.swap();
					previousProperty.compileSet(b);
				}

				notNull.setLocation();

				/*lastObject = newObject*/
				b.dup();
				b.storeLocal(lastObject);

			}
			/*!lastProperty->get lastProperty->set*/
			if(it.hasNext()) {
				property.compileGet(b);
				previousProperty=property;
			} else {
				b.swap();
				property=lastProperty(propertyList);
				property.compileSet(b);
			}
		}
	}

	protected void compileGetTarget( CodeBuilder b ) {
		Iterator it = propertyList.iterator();
		while(it.hasNext()) {
			ClassProperty property = (ClassProperty)it.next();
			if(!it.hasNext()) break;
			property.compileGet(b);
		}
	}

	/**
	 * Find the object which relates to the last property in the chain
	 * if [gs]etValue(target) was called.
	 */
	public Object getTarget( Object target ) {
		/*Find deepest target*/
		Iterator it = propertyList.iterator();
		while(it.hasNext()) {
			ClassProperty property = (ClassProperty)it.next();
			/*Skip last entry*/
			if(!it.hasNext()) break;
			target = property.getValue( target );
		}
		return target;
	}

	/**
	 * Find the object which relates to the last property in the chain
	 * if [gs]etValue(target) was called.  Create objects with null
	 * constructor along the way if necessary.
	 */
	public Object ensureTarget( Object target, Object value ) throws IllegalArgumentException {
		Object potentialTarget;
		Iterator iter = propertyList.iterator();
		/*First in chain cannot be null*/
		if(target == null)
			throw new IllegalArgumentException("Null property cannot be the first in the chain!");
		/*Loop down the property chain*/
		while(iter.hasNext()) {
			ClassProperty property = (ClassProperty)iter.next();
			/*Skipping the last entry*/
			if(!iter.hasNext()) break;
			potentialTarget = property.getValue( target );
			/*Create if necessary*/
			if(potentialTarget == null) {
				if(value == null)
					return null;
				potentialTarget = instantiate(property.getPropertyType());
				property.setValue(target, potentialTarget);
			}
			target = potentialTarget;
		}
		return target;
	}

	/**
	 * Return the value of the property of source.
	 */
	public Object getValue(Object source) {
		return lastProperty(propertyList).getValue(getTarget(source));
	}

	/**
	 * Assign a value to the property of target.
	 */
	public void setValue(Object target, Object value) throws IllegalArgumentException/*{{{*/
	{
		if(isDynamicNullHandling()) {
			target = ensureTarget(target,value);
			/*If ensureTarget returns null, that's just fine, it means
			  we're setting a null on a null, so we do nothing*/
			if(target != null)
				lastProperty(propertyList).setValue(target, value);
		} else
			lastProperty(propertyList).setValue(getTarget(target), value);
	}
}

