/*
 * Java properties library
 * Copyright (C) 2003 Jon Siddle <jon@trapdoor.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.trapdoor.properties;

import java.io.*;

import org.apache.log4j.Logger;

import cojen.classfile.*;

/**
 * Manages access to ClassProperty s.
 */
public class PropertyManager {
	private static Logger logger = Logger.getLogger(PropertyManager.class);

	/*{{{Property modes*/
	public static final int RUNTIME = 0;		/*Never compile ClassPropertys, do not attempt to load classfiles*/
	public static final int COMPILED_RO = 1;	/*Load precompiled properties, but do not compile new ones*/
	public static final int COMPILED_RW = 2;	/*Load precompiled properties and compile new ones*/
	/*}}}*/

	private static int propertyMode=COMPILED_RW;

	/**
	 * Set the current property mode.
	 * <ul>
	 * <li><b>RUNTIME</b>		Never compile ClassPropertys, do not attempt to load classfiles</li>
	 * <li><b>COMPILED_RO</b>	Load precompiled properties, but do not compile new ones</li>
	 * <li><b>COMPILED_RW</b>	Load precompiled properties and compile new ones</li>
	 * </ul>
	 */
	public static void setPropertyMode(int mode) {
		propertyMode=mode;
	}

	/**
	 * Return the current property mode.
	 * <ul>
	 * <li><b>RUNTIME</b>		Never compile ClassPropertys, do not attempt to load classfiles</li>
	 * <li><b>COMPILED_RO</b>	Load precompiled properties, but do not compile new ones</li>
	 * <li><b>COMPILED_RW</b>	Load precompiled properties and compile new ones</li>
	 * </ul>
	 */
	public static int getPropertyMode() {
		return propertyMode;
	}

	/**
	 * Return the property identified by the parameters.
	 * Creates a ClassPropertyKey from the parameters and calls getProperty(key).
	 * {@see getProperty(ClassPropertyKey)}
	 */
	public static ClassProperty getProperty(Class declaringClass, String propertyName, boolean isDynamicNullHandling) {
		return getProperty(new ClassPropertyKey(declaringClass, propertyName, isDynamicNullHandling));

	}
	/**
	 * Return the property identified by the ClassPropertyKey key.
	 * If propertyMode==COMPILED_RW, properties will be generated if not already generated.
	 * If propertyMode==RUNTIME, pre-generated properties will be ignored.
	 */
	public static ClassProperty getProperty(ClassPropertyKey key) {
		try {
			Object o;
			if(propertyMode==RUNTIME)
				return createProperty((ClassPropertyKey)key);

			o = DynamicClassManager.getObject(key);
			if(o!=null) return (ClassProperty)o;

			switch(propertyMode) {
				case COMPILED_RW:
					/*Generate object and save*/
					String className = "org.trapdoor.properties.CompiledProperty$";
					int id=DynamicClassManager.nextId(className);
					className=className+id;
					File f = DynamicClassManager.fileForClass(className);
					if(f.exists()) throw new IOException("File exists "+f.getAbsolutePath());
					OutputStream out = new FileOutputStream(f);
					try{
						writeClassForKey(key,className,out);
						out.close();
						/*Re-read saved object*/
						try {
							DynamicClassManager.loadClass(className);
							o = DynamicClassManager.getObject(key);
						} catch(ClassNotFoundException e) {
							o=null;
						}
						if(o==null) throw new IOException("Generated Class could not be re-read");
					//TODO - check what is actually thrown!
					//}catch(IllegalArgumentException e){
					}catch(IOException e){
						throw e;
					}catch(Exception e){
						logger.debug(e);
						//logger.error("Property '"+key+"' could not be compiled. Trying to use runtime property. This will probably fail!");
						out.close();
						f.delete();
						return createProperty((ClassPropertyKey)key);
					}

					break;

				case COMPILED_RO:
					o=createProperty((ClassPropertyKey)key);
					break;

				default: throw new AssertionError("Invalid COMPILATION_MODE");
			}
			return (ClassProperty)o;
		} catch(IOException e) {
			throw new IllegalStateException(e.getMessage());
		}
	}

	/*TODO - Cache.
	 * We can still improve performance by caching properties within a single execution.
	 * For the moment, this is left out to keep things simple. Performance is MUCH better
	 * with compiled properties anyway.
	 */
	/**
	 * Create a runtime property representing the property for key.
	 * Note this method may return the same object on subsequent calls, but it will
	 * never return a CompiledProperty.
	 */
	protected static ClassProperty createProperty(ClassPropertyKey key) {
		ClassProperty prop = NestedClassProperty.createProperty(key);
		return prop;
	}

	/**
	 * Convenience method to compile a call to System.out.println() with a static String.
	 */
	protected static void compilePrintln(CodeBuilder b, String s) {
		TypeDesc sysTD = TypeDesc.forClass(System.class);
		TypeDesc outTD = TypeDesc.forClass(System.out.getClass());
		b.loadStaticField(sysTD,"out",outTD);
		b.loadConstant(s);
		b.invokeVirtual(outTD,"println",null,new TypeDesc[]{TypeDesc.STRING});
	}

	/**
	 * Convenience method to compile a call to System.out.println().
	 * Expects the (presumably dynamically created, otherwise see {@see compilePrintln(CodeBuilder,String)})
	 * String to be on the TOS.
	 */
	protected static void compilePrintln(CodeBuilder b) {
		TypeDesc sysTD = TypeDesc.forClass(System.class);
		TypeDesc outTD = TypeDesc.forClass(System.out.getClass());
		b.loadStaticField(sysTD,"out",outTD);
		b.swap();
		b.invokeVirtual(outTD,"println",null,new TypeDesc[]{TypeDesc.STRING});
	}

	/**
	 * b.loadConstant(Class c) only works &gt;=1.5.
	 * This method does the equivalent.
	 */
	protected static void loadClass(CodeBuilder b, Class klass) {
		TypeDesc classTD = TypeDesc.forClass(Class.class);
		if(klass.isPrimitive()){
			/*We are somewhat "lucky" all the primitive type wrappers have a TYPE field (with the same name)*/
			b.loadStaticField(TypeDesc.forClass(klass).toObjectType(), "TYPE", classTD);
		}else{
			b.loadConstant(klass.getName());
			b.invokeStatic(classTD,"forName",classTD,new TypeDesc[]{TypeDesc.STRING});
		}
	}

	/**
	 * Generate a ClassProperty for key and write it to out.
	 * Note - we really do need className and out despite the apparent redundancy, since to build the class we
	 * need to know its name, but we should not care about where it's being stored here.
	 */
	protected static void writeClassForKey(ClassPropertyKey key, String className, OutputStream out) throws IOException {
		//System.err.println("Compiling "+oKey+" to "+className);
		ClassProperty property = createProperty(key);

		ClassFile cf = new ClassFile(className,CompiledProperty.class);

		/*Types{{{*/
		TypeDesc dcmTD = TypeDesc.forClass(DynamicClassManager.class);
		TypeDesc cpTD = TypeDesc.forClass(CompiledProperty.class);
		TypeDesc cpkTD = TypeDesc.forClass(ClassPropertyKey.class);
		TypeDesc classTD = TypeDesc.forClass(Class.class);
		TypeDesc cbTD = TypeDesc.forClass(CodeBuilder.class);
		TypeDesc exTD = TypeDesc.forClass(IllegalStateException.class);
		/*}}}*/

		cf.addField(Modifiers.PRIVATE.toStatic(true),"key",cpkTD);
		MethodInfo mi = cf.addConstructor(Modifiers.PUBLIC,
										  new TypeDesc[]{cpkTD,TypeDesc.INT,classTD,TypeDesc.BOOLEAN});
		CodeBuilder b = new CodeBuilder(mi);
		b.loadThis();
		for(int i=0; i<b.getParameterCount(); i++)
			b.loadLocal(b.getParameter(i));
		b.invokeSuperConstructor(new TypeDesc[]{cpkTD,TypeDesc.INT,classTD,TypeDesc.BOOLEAN});
		b.returnVoid();

		/*
		 * {
		 *   registerObject(
		 *	(key = new ClassPropertyKey(Class.forName("..."),"...",...))
		 *	,new this());
		 * }
		 */
		mi = cf.addInitializer();
		b = new CodeBuilder(mi);
		/*Reconstruct Key*/
		b.newObject(cpkTD);
		b.dup();
		b.dup();
		loadClass(b,key.getDeclaringClass());
		b.loadConstant(key.getPropertyName());
		b.loadConstant(key.isDynamicNullHandling());
		b.invokeConstructor(
			TypeDesc.forClass(ClassPropertyKey.class),
			new TypeDesc[]{classTD,TypeDesc.STRING,TypeDesc.BOOLEAN}
		);
		/*Save key to static field*/
		b.storeStaticField("key",cpkTD);
		/*new CompiledProperty(key,access,propertyType,isStatic)*/
		b.newObject(cf.getType());
		b.dup();
		b.loadStaticField("key",cpkTD);	/*ClassPropertyKey key*/
		b.loadConstant(property.getAccessMode());/*int access*/
		loadClass(b,property.getPropertyType());
		b.loadConstant(property.isStatic());/*boolean staticProperty*/
		b.invokeConstructor(new TypeDesc[]{cpkTD,TypeDesc.INT,classTD,TypeDesc.BOOLEAN});
		b.invokeStatic(cpTD,"registerObject",null,new TypeDesc[]{TypeDesc.OBJECT,TypeDesc.OBJECT});
		b.returnVoid();

		mi = cf.addMethod(Modifiers.PUBLIC_STATIC, "main", null,new TypeDesc[]{TypeDesc.STRING.toArrayType()});
		b = new CodeBuilder(mi);
		b.invokeStatic("getKey",cpkTD,null);
		b.invokeVirtual(TypeDesc.OBJECT, "toString", TypeDesc.STRING, null);
		compilePrintln(b);
		b.returnVoid();

		mi = cf.addMethod(Modifiers.PUBLIC_STATIC, "getKey", cpkTD,null);
		b = new CodeBuilder(mi);
		b.loadStaticField("key",cpkTD);
		b.returnValue(cpkTD);

		mi = cf.addMethod(Modifiers.PROTECTED, "compileGet", null,new TypeDesc[]{cbTD});
		b = new CodeBuilder(mi);
		b.newObject(exTD);
		b.dup();
		b.loadConstant("Operation not supported - property already compiled.");
		b.invokeConstructor(exTD,new TypeDesc[]{TypeDesc.STRING});
		b.throwObject();
		b.returnVoid();

		mi = cf.addMethod(Modifiers.PROTECTED, "compileSet", null,new TypeDesc[]{cbTD});
		b = new CodeBuilder(mi);
		b.newObject(exTD);
		b.dup();
		b.loadConstant("Operation not supported - property already compiled.");
		b.invokeConstructor(exTD,new TypeDesc[]{TypeDesc.STRING});
		b.throwObject();
		b.returnVoid();

		mi = cf.addMethod(Modifiers.PROTECTED, "compileGetTarget", null,new TypeDesc[]{cbTD});
		b = new CodeBuilder(mi);
		b.newObject(exTD);
		b.dup();
		b.loadConstant("Operation not supported - property already compiled.");
		b.invokeConstructor(exTD,new TypeDesc[]{TypeDesc.STRING});
		b.throwObject();
		b.returnVoid();

		mi = cf.addMethod(Modifiers.PUBLIC, "getValue", TypeDesc.OBJECT,new TypeDesc[]{TypeDesc.OBJECT});
		b = new CodeBuilder(mi);
		b.loadLocal(b.getParameter(0));
		property.compileGet(b);
		b.returnValue(TypeDesc.OBJECT);

		mi = cf.addMethod(Modifiers.PUBLIC, "setValue", null, new TypeDesc[]{TypeDesc.OBJECT,TypeDesc.OBJECT});
		b = new CodeBuilder(mi);
		b.loadLocal(b.getParameter(0));
		b.loadLocal(b.getParameter(1));
		property.compileSet(b);
		b.returnVoid();

		mi = cf.addMethod(Modifiers.PUBLIC, "getTarget", TypeDesc.OBJECT,new TypeDesc[]{TypeDesc.OBJECT});
		b = new CodeBuilder(mi);
		b.loadLocal(b.getParameter(0));
		property.compileGetTarget(b);
		b.returnValue(TypeDesc.OBJECT);

		cf.writeTo(out);
	}
}
