/*
 * Java properties library
 * Copyright (C) 2003 Jon Siddle <jon@trapdoor.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.trapdoor.properties;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

import cojen.classfile.*;

/**
 * Simple class property.
 * This ClassProperty implementation defines properties in terms of
 * a pair of methods; one accessor and one mutator method.
 * 
 */
public class SimpleClassProperty extends ClassProperty {
	private Method accessor;
	private Method mutator;

	/**
	 * Construct a simple property relating to declaring class.
	 * The (declaring class, propertyName) tuple determines a
	 * "unique" property. accessor and mutator must both be methods
	 * of declaringClass (or its ancestor classes).
	 * Property names have class scope.
	 * This constructor should not be used directly except in exceptional
	 * circumstances. The getProperty methods should be used instead.
	 * It is public since this allows for exceptional circumstances,
	 * getProperty is preferred since it checks the cache first. This
	 * constructor will of course add the new property to the cache,
	 * overwriting a property with the same property key.
	 * The property access and static attributes are determined from the
	 * methods passed.
	 * @param declaringClass The class with which the property should be associated.
	 * @param propertyName The namne of the property.
	 * @param accessor The accessor method (usually getXXX). null for write-only (CP_WO) properties.
	 * @param mutator The mutator method (usually setXXX). null for read-only (CP_RO) properties.
	 */
	private SimpleClassProperty(
		Class declaringClass, String propertyName, int access, Class propertyType, boolean isStatic,
		boolean isDynamicNullHandling, Method accessor, Method mutator
	) throws IllegalArgumentException/*{{{*/
	{
		/*Ideally we would throw an exception if accessor and mutator are both null
		  to avoid this ugliness, but the super constructor must be the first call.*/
		super(declaringClass, propertyName, access, propertyType, isStatic, isDynamicNullHandling);

		/*Check class is accessable*/
		if(!Modifier.isPublic(declaringClass.getModifiers()))
			throw new IllegalArgumentException("'"+declaringClass.getName()+"' is not public.");

		/*Check accessor/mutator presence*/
		if(accessor==null && mutator==null)
			throw new IllegalArgumentException("Cannot create a property without accessor or mutator");

		/*check accessor/mutator belong to class*/
		Class ascendingClass = declaringClass;
		boolean foundAccessor=false, foundMutator=false;
		while(ascendingClass!=null) {
			if(accessor!=null && accessor.getDeclaringClass()==ascendingClass) foundAccessor=true;
			if(mutator!=null && mutator.getDeclaringClass()==ascendingClass) foundMutator=true;
			if(foundAccessor && foundMutator) break;
			ascendingClass=ascendingClass.getSuperclass();
		}
		if( (accessor!=null && !foundAccessor) || (mutator!=null && !foundMutator) )
			throw new IllegalArgumentException("Accessor and Mutator must belong to declaring class.");

		/*Check that accessor/mutator static modifiers match*/
		if( accessor!=null && mutator!=null &&
				( Modifier.isStatic(accessor.getModifiers())!=Modifier.isStatic(mutator.getModifiers()) )
		  )
			throw new IllegalArgumentException("Static modifiers do not match for accessor and mutator");

		/*Check that static specification is correct*/
		if( Modifier.isStatic( ((accessor==null)?mutator:accessor).getModifiers() )!=isStatic )
			throw new IllegalArgumentException("Static modifiers do not match specification");

		/*Check method access*/
		/*
		if( accessor!=null && !Modifier.isPublic(accessor.getModifiers()) )
			throw new IllegalArgumentException("'"+accessor.getName()+"' is not public.");
		if( mutator!=null && !Modifier.isPublic(mutator.getModifiers()) )
			throw new IllegalArgumentException("'"+mutator.getName()+"' is not public.");
			*/

		this.accessor=accessor;
		this.mutator=mutator;

	}

	public static ClassProperty createProperty(ClassPropertyKey key) {
		return createProperty(key.getDeclaringClass(), key.getPropertyName(), key.isDynamicNullHandling());
	}

	/**
	 * Wrapper around full getProperty() which uses default accessor/mutator names and property type.
	 * The accessor and mutator names are assumed to be <code>"get"+propertyName</code>
	 * and </code>"set"+propertyName</code> respectively.
	 * The property type is determined from the accessor method's return value and
	 * so the property cannot be write-only (use the full getProperty for this case).
	 */
	public static ClassProperty createProperty(Class declaringClass, String propertyName, boolean isDynamicNullHandling ) {
		return createProperty( declaringClass, propertyName, isDynamicNullHandling, null, null, null );
	}

	/**
	 * Create a property given the class, property name and optioanlly overriding the name of the
	 * accessor, mutator methods, and property type.
	 * If a property with the same property key already exists, it is returned instead, since
	 * properties should be constant this is simply caching, and vastly improves efficiency.
	 */
	public static ClassProperty createProperty(/*{{{*/
		Class declaringClass, String propertyName, boolean isDynamicNullHandling, Class propertyType,
		String accessorName, String mutatorName ) {
		Method accessor=null;
		Method mutator=null;
		boolean defaultAccessorName=false;
		boolean defaultMutatorName=false;

		if(accessorName==null) {
			accessorName="get"+propertyName; defaultAccessorName=true;
		}
		if(mutatorName==null) {
			mutatorName="set"+propertyName; defaultMutatorName=true;
		}

		//System.err.println("Searching for "+accessorName+" in "+declaringClass.getName());

		/*Ascend class heirarchy, looking for the accessor/mutator definitions.
		  NOTE we have to use 2 loops since the mutator method argument must
		  match the property type which is usually determined by the return
		  type of the accessor method.
		*/
		Class ascendingClass=declaringClass;
		while( ascendingClass!=null && accessor==null ) {
			/*Try to find accessor, CNFEx causes accessor==null*/
			try {
				accessor=ascendingClass.getDeclaredMethod(accessorName, null);
			} catch( NoSuchMethodException eGet ) {
				//System.err.println(accessorName+" not declared on "+ascendingClass.getName());
				if( defaultAccessorName ){
					String altAccessorName = "is"+propertyName;
					try {
						accessor=ascendingClass.getDeclaredMethod(altAccessorName, null);
					} catch( NoSuchMethodException eIs ) {
						//System.err.println(altAccessorName+" not declared on "+ascendingClass.getName());
					}
				}
			}

			/*If accessor not found, try superclass*/
			if(accessor==null) ascendingClass=ascendingClass.getSuperclass();
			/*If type is unspecified, and accessor was found, set from return type*/
			else if(propertyType==null) propertyType=accessor.getReturnType();
		}

		if( propertyType==null )
			throw new IllegalArgumentException("No type specified, and accessor method not found");

		ascendingClass=declaringClass;
		while( ascendingClass!=null && mutator==null ) {
			try {
				mutator=ascendingClass.getDeclaredMethod(mutatorName, new Class[]{propertyType});
			} catch( NoSuchMethodException e ) {
				//System.err.println(mutatorName+" not declared on "+ascendingClass.getName());
				ascendingClass=ascendingClass.getSuperclass();
			}
		}


		if( accessor==null && mutator==null )
			throw new IllegalArgumentException("Neither accessor nor mutator found");

		/* Override Access Control. FIXME - This functionality should be explicitly requested. */
		if(accessor!=null){
			//logger.warn("Overriding access control");
			if(!Modifier.isPublic(accessor.getModifiers())) accessor.setAccessible(true);
		}
		if(mutator!=null){
			//logger.warn("Overriding access control");
			if(!Modifier.isPublic(mutator.getModifiers())) mutator.setAccessible(true);
		}

		int accessMode = (accessor==null)?CP_WO:((mutator==null)?CP_RO:CP_RW);
		boolean isStatic = (accessor==null)?
						   Modifier.isStatic(mutator.getModifiers()):
						   Modifier.isStatic(accessor.getModifiers());
		return new SimpleClassProperty(
				   declaringClass,propertyName,accessMode,propertyType,isStatic,isDynamicNullHandling,
				   accessor,mutator);
	}

	protected void compileGetTarget( CodeBuilder b ) {}

	protected void compileGet( CodeBuilder b ) {
		TypeDesc classTD = TypeDesc.forClass(Class.class);
		TypeDesc exTD = TypeDesc.forClass(IllegalStateException.class);

		if(accessor==null) {
			b.newObject(exTD);
			b.dup();
			b.loadConstant("getValue() called on write-only property");
			b.loadThis();
			b.invokeVirtual("toString",TypeDesc.STRING, null);
			b.invokeVirtual(TypeDesc.STRING,"concat",TypeDesc.STRING,new TypeDesc[]{TypeDesc.STRING});
			b.invokeConstructor(exTD,new TypeDesc[]{TypeDesc.STRING});
			b.throwObject();
		} else {
			TypeDesc dclassTD = TypeDesc.forClass(accessor.getDeclaringClass());
			TypeDesc valueTD = TypeDesc.forClass(accessor.getReturnType());

			/*Check source!=null*/
			b.dup();
			Label notNull = b.createLabel();
			Label afterNull = b.createLabel();
			b.ifNullBranch(notNull,false);
			if(isDynamicNullHandling()) {
				b.pop();
				b.loadNull();
				b.branch(afterNull);
			} else {
				b.newObject(exTD);
				b.dup();
				b.loadConstant("cannot de-reference null property calling "+accessor.getName());
				b.loadThis();
				b.invokeVirtual("toString",TypeDesc.STRING, null);
				b.invokeVirtual(TypeDesc.STRING,"concat",TypeDesc.STRING,new TypeDesc[]{TypeDesc.STRING});
				b.invokeConstructor(exTD,new TypeDesc[]{TypeDesc.STRING});
				b.throwObject();
			}
			notNull.setLocation();

			/*Check source instanceof declaringclass*/
			b.dup();
			b.instanceOf(dclassTD);
			Label assignable = b.createLabel();
			b.ifZeroComparisonBranch(assignable,"!=");
			b.loadConstant("DeclaringClass="+getDeclaringClass().getName()
						   +" but ObjectClass=");
			b.swap();
			b.invokeVirtual(TypeDesc.OBJECT,"getClass",classTD,null);
			b.invokeVirtual(classTD,"getName",TypeDesc.STRING,null);
			b.invokeVirtual(TypeDesc.STRING,"concat",TypeDesc.STRING,new TypeDesc[]{TypeDesc.STRING});
			LocalVariable errMsg = b.createLocalVariable("errMsg",TypeDesc.STRING);
			b.storeLocal(errMsg);
			b.newObject(exTD);
			b.dup();
			b.loadLocal(errMsg);
			b.loadThis();
			b.invokeVirtual("toString",TypeDesc.STRING, null);
			b.invokeVirtual(TypeDesc.STRING,"concat",TypeDesc.STRING,new TypeDesc[]{TypeDesc.STRING});
			b.invokeConstructor(exTD,new TypeDesc[]{TypeDesc.STRING});
			b.throwObject();
			assignable.setLocation();

			/*Cast source (won't fail since we checked above)*/
			b.checkCast(dclassTD);

			b.invokeVirtual(
				dclassTD,
				accessor.getName(),
				valueTD,
				null
			);
			/*Box primitives*/
			if(valueTD.isPrimitive()){
				b.convert(valueTD,valueTD.toObjectType());
			}
			afterNull.setLocation();
		}
	}

	protected void compileSet( CodeBuilder b ) {
		TypeDesc exTD = TypeDesc.forClass(IllegalStateException.class);

		if(mutator==null) {
			b.newObject(exTD);
			b.dup();
			b.loadConstant("setValue() called on read-only property");
			b.loadThis();
			b.invokeVirtual("toString",TypeDesc.STRING, null);
			b.invokeVirtual(TypeDesc.STRING,"concat",TypeDesc.STRING,new TypeDesc[]{TypeDesc.STRING});
			b.invokeConstructor(exTD,new TypeDesc[]{TypeDesc.STRING});
			b.throwObject();
		} else {
			TypeDesc classTD = TypeDesc.forClass(Class.class);
			TypeDesc dclassTD = TypeDesc.forClass(mutator.getDeclaringClass());
			TypeDesc valueTD = TypeDesc.forClass(mutator.getParameterTypes()[0]);
			TypeDesc valueObjTD = valueTD;

			b.swap();

			/*Check target!=null*/
			b.dup();
			Label notNull = b.createLabel();
			b.ifNullBranch(notNull,false);
			b.newObject(exTD);
			b.dup();
			b.loadConstant("cannot de-reference null property calling "+mutator.getName());
			b.loadThis();
			b.invokeVirtual("toString",TypeDesc.STRING, null);
			b.invokeVirtual(TypeDesc.STRING,"concat",TypeDesc.STRING,new TypeDesc[]{TypeDesc.STRING});
			b.invokeConstructor(exTD,new TypeDesc[]{TypeDesc.STRING});
			b.throwObject();
			notNull.setLocation();

			/*Check target instanceof declaringclass*/
			b.dup();
			b.instanceOf(dclassTD);
			Label assignable = b.createLabel();
			b.ifZeroComparisonBranch(assignable,"!=");
			b.loadConstant("DeclaringClass="+getDeclaringClass().getName()
						   +" but ObjectClass=");
			b.swap();
			b.invokeVirtual(TypeDesc.OBJECT,"getClass",classTD,null);
			b.invokeVirtual(classTD,"getName",TypeDesc.STRING,null);
			b.invokeVirtual(TypeDesc.STRING,"concat",TypeDesc.STRING,new TypeDesc[]{TypeDesc.STRING});
			LocalVariable errMsg = b.createLocalVariable("errMsg",TypeDesc.STRING);
			b.storeLocal(errMsg);
			b.newObject(exTD);
			b.dup();
			b.loadLocal(errMsg);
			b.loadThis();
			b.invokeVirtual("toString",TypeDesc.STRING, null);
			b.invokeVirtual(TypeDesc.STRING,"concat",TypeDesc.STRING,new TypeDesc[]{TypeDesc.STRING});
			b.invokeConstructor(exTD,new TypeDesc[]{TypeDesc.STRING});
			b.throwObject();
			assignable.setLocation();

			/*Cast target (won't fail since we checked above)*/
			b.checkCast(dclassTD);

			b.swap();

			/*If the type is primitive, use the box type to compare*/
			if(valueTD.isPrimitive()){
				valueObjTD = valueTD.toObjectType();
			}

			/*Check value instanceof valueclass*/
			Label typeOk = b.createLabel();
			Label typeFail = b.createLabel();

			b.dup();

			/* For primitive types, null is automatically failed,
			 * for object types, null is automatically passed.
			 */
			if(valueTD.isPrimitive()){
				b.ifNullBranch(typeFail,true);
			}else{
				b.ifNullBranch(typeOk,true);
			}

			b.dup();
			b.instanceOf(valueObjTD);
			b.ifZeroComparisonBranch(typeOk,"!=");

			typeFail.setLocation();
			b.loadConstant("PropertyType="+getPropertyType().getName()+" but ObjectType=");
			b.swap();
			b.invokeVirtual(TypeDesc.OBJECT,"getClass",classTD,null);
			b.invokeVirtual(classTD,"getName",TypeDesc.STRING,null);
			b.invokeVirtual(TypeDesc.STRING,"concat",TypeDesc.STRING,new TypeDesc[]{TypeDesc.STRING});
			errMsg = b.createLocalVariable("errMsg",TypeDesc.STRING);
			b.storeLocal(errMsg);
			b.newObject(exTD);
			b.dup();
			b.loadLocal(errMsg);
			b.loadThis();
			b.invokeVirtual("toString",TypeDesc.STRING, null);
			b.invokeVirtual(TypeDesc.STRING,"concat",TypeDesc.STRING,new TypeDesc[]{TypeDesc.STRING});
			b.invokeConstructor(exTD,new TypeDesc[]{TypeDesc.STRING});
			b.throwObject();

			typeOk.setLocation();

			/*Cast value (won't fail since we checked above).*/
			b.checkCast(valueObjTD);

			if(valueTD.isPrimitive()){
				/*Unbox primitive*/
				b.convert(valueObjTD, valueTD);
			}

			b.invokeVirtual(
				dclassTD,
				mutator.getName(),
				null,
				new TypeDesc[]{valueTD}
			);
		}
	}


	/**
	 * Return the value of the property of source.
	 */
	public Object getValue(Object source) {
		if(accessor==null) throw new IllegalStateException("Tried to read write-only property '"+getPropertyName());
		if(source==null) {
			if(isDynamicNullHandling())
				return null;
			else
				throw new IllegalStateException("Tried to read property from null object ("+accessor+")");
		}
		if(!getDeclaringClass().isAssignableFrom(source.getClass()))
			throw new IllegalStateException("Tried to read property from incompatible object.");
		try {
			return accessor.invoke(source, new Object[]{});
		} catch(InvocationTargetException e) {
			if(e.getCause() instanceof IllegalArgumentException)
				throw (IllegalArgumentException)e.getCause();
			AssertionError error = new AssertionError("Accessor " +accessor+ " threw exception: "+e.getCause());
			error.initCause(e);
			throw error;

		} catch(IllegalAccessException e) {
			throw new AssertionError("Accessor "+accessor+"threw access exception, "+
									 "but this should already have been checked!");
		}
	}

	/**
	 * Assign a value to the property of target.
	 */
	public void setValue(Object target, Object value) throws IllegalArgumentException/*{{{*/
	{
		if(mutator==null) throw new IllegalStateException("Tried to write read-only property '"+getPropertyName());
		if(target==null) throw new IllegalStateException("Tried to write property to null object.");
		if(!getDeclaringClass().isAssignableFrom(target.getClass()))
			throw new IllegalStateException("Tried to write property to incompatible object.");
		try {
			mutator.invoke(target, new Object[]{value});
		} catch(InvocationTargetException e) {
			if(e.getCause() instanceof IllegalArgumentException)
				throw (IllegalArgumentException)e.getCause();
		} catch(IllegalAccessException e) {
			throw new AssertionError("Mutator threw access exception, "+
									 "but this should already have been checked!");
		}
	}

	public Object getTarget(Object target) {
		return target;
	}
}


