/*
 * Java properties library
 * Copyright (C) 2003 Jon Siddle <jon@trapdoor.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package test;

import junit.framework.TestCase;
import junit.framework.Test;
import junit.framework.TestSuite;

import org.trapdoor.properties.*;

import java.io.File;

public class CompileTest extends TestCase {
	public static class A {
		private B b;
		public void setB(B b) { this.b = b; }
		public B getB() { return b; }
	}

	public static class B {
		private C c;
		public void setC(C c) { this.c = c; }
		public C getC() { return c; }
	}

	public static class C {
		private String d;
		public void setD(String d) { this.d = d; }
		public String getD() { return d; }
	}

	public void setUp() throws ClassNotFoundException {
		File tmpdir = new File("test-precompiled");
		if(tmpdir.exists()) tmpdir.delete();
		tmpdir.mkdir();
		DynamicClassManager.purge();
		DynamicClassManager.init(tmpdir);
		DynamicClassManager.load();
		PropertyManager.setPropertyMode(PropertyManager.COMPILED_RW);
	}

	public void testCompile() {
		A a = new A();
		a.setB(new B());
		a.getB().setC(new C());
		a.getB().getC().setD("Test String");

		ClassPropertyKey key = new ClassPropertyKey(A.class, "B:C:D", false);
		ClassProperty p = PropertyManager.getProperty(key);

		assertTrue(p instanceof CompiledProperty);
		assertEquals("test-precompiled/org/trapdoor/properties/CompiledProperty$1.class",DynamicClassManager.fileForClass(p.getClass().getName()).getPath());
	}

	public void testRuntime() {
		PropertyManager.setPropertyMode(PropertyManager.RUNTIME);

		ClassPropertyKey key = new ClassPropertyKey(A.class, "B:C:D", false);
		ClassProperty p = PropertyManager.getProperty(key);
		assertEquals("test-precompiled/org/trapdoor/properties/NestedClassProperty.class",DynamicClassManager.fileForClass(p.getClass().getName()).getPath());
	}

	public void testReload() throws ClassNotFoundException {
		A a = new A();
		a.setB(new B());
		a.getB().setC(new C());
		a.getB().getC().setD("Test String");

		ClassPropertyKey key = new ClassPropertyKey(A.class, "B:C:D", false);
		ClassProperty p = PropertyManager.getProperty(key);

		assertTrue(p instanceof CompiledProperty);
		assertEquals("test-precompiled/org/trapdoor/properties/CompiledProperty$1.class",DynamicClassManager.fileForClass(p.getClass().getName()).getPath());

		ClassPropertyKey key2 = new ClassPropertyKey(A.class, "B:C:D", false);
		ClassProperty p2 = PropertyManager.getProperty(key);
		assertTrue(p==p2);

		DynamicClassManager.purge();
		DynamicClassManager.load();

		ClassPropertyKey key3 = new ClassPropertyKey(A.class, "B:C:D", false);
		ClassProperty p3 = PropertyManager.getProperty(key);
		assertFalse(p==p3);
		assertEquals("test-precompiled/org/trapdoor/properties/CompiledProperty$1.class",DynamicClassManager.fileForClass(p3.getClass().getName()).getPath());

	}

        public static Test suite() {
                return new TestSuite(CompileTest.class);
        }
}
