/*
 * Java properties library
 * Copyright (C) 2003 Jon Siddle <jon@trapdoor.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package test;

import junit.framework.TestCase;
import junit.framework.Test;
import junit.framework.TestSuite;

import org.trapdoor.properties.*;

public class NestedPropertyTest extends TestCase {
	public static class Outer {
		private Inner a;
		public void setA(Inner a) { this.a = a; }
		public Inner getA() { return a; }
	}

	public static class Inner {
		private String theString;
		public void setTheString(String theString) { this.theString = theString; }
		public String getTheString() { return theString; }
	}

	/* Test that a nested property degrades gracefully */
        public void testNestedSimple() {
		ClassProperty p = NestedClassProperty.createProperty(Outer.class,"A",false);
		assertTrue(p instanceof NestedClassProperty);
		assertTrue(p.getTarget(new Outer()) instanceof Outer);
		assertEquals("A", p.getPropertyName());
		assertEquals(Outer.class, p.getDeclaringClass());
		assertEquals(Inner.class, p.getPropertyType());
		assertFalse(p.isStatic());
	}

	/* Test the basic properties of a nested property */
        public void testNested() {
		ClassProperty p = NestedClassProperty.createProperty(Outer.class,"A:TheString",false);
		assertTrue(p instanceof NestedClassProperty);

		Outer o = new Outer();
		assertNull(p.getTarget(o));
		o.setA(new Inner());
		assertTrue(p.getTarget(o) instanceof Inner);

		assertEquals("A:TheString", p.getPropertyName());
		assertEquals(Outer.class, p.getDeclaringClass());
		assertEquals(String.class, p.getPropertyType());
		assertFalse(p.isStatic());
	}

	/* Test basic value manipulation */
        public void testPropertyValueManipulation() {
		Outer o = new Outer();
		o.setA(new Inner());
		o.getA().setTheString("Test String");

		ClassProperty p = NestedClassProperty.createProperty(Outer.class,"A:TheString",false);

		assertEquals("Test String", p.getValue(o));
		p.setValue(o,"Another test string");
		assertEquals("Another test string", p.getValue(o));
        }

	public void testDynamicNull() {
		ClassProperty p = NestedClassProperty.createProperty(Outer.class,"A:TheString",true);
		assertNull(p.getTarget(null));
		assertNull(p.getValue(null));

		Outer o = new Outer();
		assertNull(p.getTarget(o));
		assertNull(p.getValue(o));

		Inner i = new Inner();
		o.setA(i);
		assertEquals(i,p.getTarget(o));
		assertNull(p.getValue(o));
	}


	public void testDynamicNullSet() {
		Outer o = new Outer();
		ClassProperty p = NestedClassProperty.createProperty(Outer.class,"A:TheString",true);
		p.setValue(o,"Test String");
		assertEquals("Test String", p.getValue(o));
	}

        public static Test suite() {
                return new TestSuite(NestedPropertyTest.class);
        }
}

